/************************************************************************
 * NAME:	tosvd
 *
 * DESCR:	"Super converter" that converts files from and to the SVD
 *		format, as well as between many other formats.
 *
 *		It operates in "reporting mode" too, such that it can be
 *		used to report on the file format of the incoming files.
 *			
 *		Incoming formats are guessed, but can be forced
 *		if desired or necessary.
 *
 *		See Usage at bottom.
 *
 *		The conversion is sent to standard out.
 *
 * NOTE - the -o option is not yet implemented...in fact the USAGE is
 *	  wrong for the way in which output is generated.
 ************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>

#include "standard.h"
#include "error.h"
#include "floppy.h"

/************************************************************************/
/*		  RELEASE VERSION NUMBER GOES HERE!			*/
/************************************************************************/

#define VERSION	"2.2"

main(argc,argv)
int	argc;
char	*argv[];
{
    int			ch;
    int			fd;
    struct floppy	floppy;
    int			just_guess = FALSE;	/* don't generate svd	*/
    int			check = FALSE;		/* check CRC		*/
    int			set_wp = FALSE;		/* write protect set	*/
    int			wp = FALSE;		/* write protect setting*/
    int			force_CRC = FALSE;	/* force CRC write	*/
    int			verbose = FALSE;	/* level of verbosity	*/
    int			terse = FALSE;		/* level of verbosity	*/
    int			report = FALSE;		/* report disk params	*/
    int			ignore_CRC = FALSE;	/* ignore CRC errors	*/
    int			machineform = FALSE;	/* machine readable form*/
    char		*format_id = NULL;
    char		*force_format_id = NULL;
    char		*force_format_params = NULL;
    char		*output_format_id = NULL;
    int			 input_format_query = FALSE;
    int			 output_format_query = FALSE;
    int			 extension_query = FALSE;

  (void)floppy_init(&floppy);

  floppy_format_init(); 	/* initialize all supported disk formats */

  opterr = 0;

  for (;;) {
    ch = getopt(argc, argv, "?gcw:Cvrs:o:itmVQqe");
    if (ch == -1) break;
    switch (ch) {
    case 'e':
	extension_query = TRUE;
	break;
    case 'i':
	ignore_CRC = TRUE;
	break;
    case 'r':
	report = TRUE;
	break;
    case 'v':
	/* may want to turn this into a number	*/
	verbose = TRUE;
	break;
    case 't':
	terse = TRUE;
	break;
    case 'm':
	machineform = TRUE;
	break;
    case 'w':
	if (strcmp(argv[optind-1],"on") == 0 || strcmp(argv[optind-1],"1") == 0) {
	    set_wp = TRUE;
	    wp = TRUE;
	} else if (strcmp(argv[optind-1],"off") == 0 || strcmp(argv[optind-1],"0") == 0) {
	    set_wp = TRUE;
	    wp = FALSE;
	} else {
	    fprintf(stderr,"%s: specify either on|off or 1|0 for the -w option.\n",argv[0]);
	    usage(argv[0]);
	    exit(3);
	}
	break;
    case 'c':
	check = TRUE;
	break;
    case 'C':
	force_CRC = TRUE;
	break;
    case 's':
	force_format_id = floppy_format_list_select(argv[optind-1],FALSE);
	force_format_params = floppy_format_add_params(argv[optind-1]);
	if (force_format_id == NULL) {
	    fprintf(stderr,"%s: -s format type \"%s\" unknown.\n",argv[0],argv[optind-1]);
	    usage(argv[0]);
	    exit(3);
	}
	break;
    case 'Q':
	input_format_query = TRUE;
	break;
    case 'q':
	output_format_query = TRUE;
	break;
    case 'g':
	just_guess = TRUE;
	break;
    case 'o':
	output_format_id = floppy_format_list_select(argv[optind-1],TRUE);
	if (output_format_id == NULL) {
	    fprintf(stderr,"%s: -o format type \"%s\" unknown or not supported.\n",argv[0],argv[optind-1]);
	    usage(argv[0]);
	    exit(3);
	}
	break;
    case 'V':
	printf("%s: version %s (%s)\n",argv[0],VERSION,COMPDATE);
	exit(0);
    case '?':
	usage(argv[0]);
	exit(0);
    default:
	usage(argv[0]);
	exit(2);
    }
  }

  /* note that the format queries cause any other options to be ignored.	*/

  if (input_format_query || output_format_query) {
      char	buffer[1000];

      if (input_format_query) {
	  floppy_format_list_string(buffer,FALSE,verbose);
	  printf("%s",buffer);
	  if (!verbose) {
	      printf("\n");
	  }
      }

      if (output_format_query) {
	  floppy_format_list_string(buffer,TRUE,verbose);
	  printf("%s",buffer);
	  if (!verbose) {
	      printf("\n");
	  }
      }
      exit(0);
  }

  if (extension_query) {
      char	buffer[1000];

      floppy_fileext_query(buffer,verbose);
      printf("%s",buffer);
      if (!verbose) {
	  printf("\n");
      }
      exit(0);
  }

  if (optind > argc - 1) {
      /* must have a filename (need to rewind it!)	*/
      fprintf(stderr,"%s: you must specify at least one filename.\n",argv[0]);
      usage(argv[0]);
      exit(2);
  }

  for( ; optind <= argc-1; optind++) {

      fd = open(argv[optind],O_RDONLY);
      if (fd < 0) {
	  fprintf(stderr,"%s: file not found \"%s\"\n",argv[0],argv[optind]);
	  exit(2);
      }

      if (force_format_id == NULL) {
	  char 	*ext;

	  ext = floppy_file_extension(argv[optind]);

	  format_id = floppy_format_guesser(fd, ext);
      }

      if (just_guess || report) {
	  int	level = (!report)?1:(!verbose)?2:3;
	  if (machineform) {
	      level = 4;
	  }

	  if (!terse && !machineform) {
	      printf("%s: %s - ",argv[0], argv[optind]);
	  }

	  if (force_format_id == NULL && format_id == NULL) {
	      printf("UNKNOWN FORMAT\n");
	  } else {
	      floppy_format_report((force_format_id==NULL)?format_id:force_format_id,fd,&floppy,level);
	  }
      } else {

	  lseek(fd,(off_t)0,SEEK_SET);

	  if (force_format_id != NULL || format_id != NULL) {
	      if (floppy_format_reader((force_format_id==NULL)?format_id:force_format_id,fd,&floppy) !=0) {
		  fprintf(stderr,"%s: error reading %s format\n",argv[0],format_id);
		  exit(1);
	      }
	  }

	  if (set_wp) {
	      floppy.write_protect = (wp)?0xff:0x00;	/* if forcing write protect	*/
	  }

	  if (force_CRC) {
	      floppy_crc_set(&floppy);
	  } else {
	      if (!ignore_CRC) {
		  if (floppy_crc_check(&floppy,1,verbose,argv[0]) != 0) {	/* check the CRCs	*/
		      fprintf(stderr,"%s: CRC error(s).  No output written.  Use -i or -C.\n",argv[0]);
		      exit (2);
		  }
	      }
	  }

	  floppy_sectormap(&floppy);

	  if (!check) {
	      if (output_format_id == NULL) {
		  output_format_id = floppy_format_list_select("SVD",TRUE);
	      }
	      (void) floppy_format_dumper(output_format_id,1,&floppy);
	  }
      }
  }

  exit(0);
}

usage(char *name)
{
  char buffer[1000];

  printf("Usage: %s [option(s)] FILENAME [FILENAMES...] > outputfile\n\n",name);

  printf("   Converts the FILENAME to SVD format.\n\n");
  printf("   -s fmt     ---  forces an input format; fmt needs to\n");
  printf("                   to be one of: %s\n", floppy_format_list_string(buffer,FALSE,FALSE));
  printf("   -c         ---  just check the disk CRC if possible\n");
  printf("   -C         ---  recalc and write CRC during process\n");
  printf("   -w on|off  ---  set the write protect flag for the output image\n");
  printf("   -v         ---  be verbose when working\n");
  printf("   -t         ---  be terse when working (used in GUI)\n");
  printf("   -m         ---  output machine readable form (used in GUI)\n");
  printf("   -i         ---  ignore CRC errors\n");
  printf("   -r         ---  only report on given disk (no svd generation)\n");
  printf("   -o fmt     ---  output is written in the given fmt; fmt must be\n");
  printf("                   one of: %s\n", floppy_format_list_string(buffer,TRUE,FALSE));
  printf("                   (defaults to: SVD)\n");
  printf("   -q         ---  query/list the output formats available\n");
  printf("   -Q         ---  query/list the input formats available\n");
  printf("   -V         ---  print out version number\n");
  printf("   -e         ---  print out list of known extensions and format mappings (in order)\n");
  printf("   -g         ---  just do a guess (no svd generation)\n\n");
}
